//*************************************************************************************************
//
//	Description:
//		hud.fx - A fork of basic.fx with support for self illumination and a global translucency
//						 dial.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship			12/07/2006	1.0						Created
//		TMann						12/11/2006	1.1						PS3 & Win_GL versions
//		TMann						01/09/2007	1.4						Added PS3/GL literal param passing
//		TNettleship			02/12/2007	1.5						Fixed environment mapping
//		AlanS						13/02/2007	1.6					  Added "Motec" technique ( self illum with no diffuse alpha mod )
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship     09/28/2007  2.03					Added specialisation.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

float specularRollOff													// Specular lighting overlighting rolloff amount
<
	string UIName = "Specular Rolloff";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.1f;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif

float globalTranslucencyFactor
<
	string UIName = "Translucency Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

bool getTransparencyFromAlpha
<
	string UIName = "Get Transparency from Alpha?";
	bool appEdit = true;
> = false;

float globalSelfIlluminationFactor
<
	string UIName = "Self Illumination Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.0f;

bool additiveSelfIllumination
<
	string UIName = "Additive SI?";
	bool appEdit = true;
> = false;


// Simplify shadowing code to access only the first (the nearest) shadowmap.
SPECIALISATION_PARAM( forceFistShadowmap, "Force the first shadowmap?", "FORCE_FIRST_SHADOWMAP" )


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 4 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT HUDVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
};

#else

struct PSINPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS( 4 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
#ifdef _POST_PARSE_
PSOUTPUT HUDFragmentShader( PSINPUT _input )
#else
PSOUTPUT HUDFragmentShader( PSINPUT _input, uniform bool _premultiplyAlpha, uniform bool _selfIllumNoAlpha )
#endif
{
	PSOUTPUT _output = ( PSOUTPUT )0;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );

	float selfIllumFactor;
	// If this is a material which should be opaque
	if ( ( !_premultiplyAlpha ) && ( !getTransparencyFromAlpha ) )
	{
		// Force diffuse alpha to 1.0 (solves problems with rearview mirror on PS3)
		diffuseTexColour.a = 1.0f;
	}

	float transFactor;

	// If this is a selfilluminated material
	if ( globalSelfIlluminationFactor > 0.0f )
	{
		// SI factor from the diffuse alpha * global factor
		if( _selfIllumNoAlpha )
		{
			selfIllumFactor = globalSelfIlluminationFactor;
		}
		else
		{
			selfIllumFactor = globalSelfIlluminationFactor * diffuseTexColour.a;
		}
	}
	// Otherwise not self illuminated
	else
	{
		// SI factor is 0.0
		selfIllumFactor = 0.0f;
	}

	if( getTransparencyFromAlpha )
	{
		// Trans factor is global trans * diffuse alpha
		transFactor = globalTranslucencyFactor * diffuseTexColour.a;
	}
	else
	{
		// Trans factor from the global trans
		transFactor = globalTranslucencyFactor;
	}

	// Albedo (posh name for diffuse colour) is diffuse.rgb * translucency
	float4 albedo;
	if ( _premultiplyAlpha )
	{
		albedo = float4( diffuseTexColour.rgb * transFactor, 1.0f );
	}
	else
	{
		albedo = float4( diffuseTexColour.rgb, 1.0f );
	}

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap.rgb = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap.rgb = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate ambient illumination
	float4 accumulatedColour = albedo * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 worldReflectionVector = reflect( -eye, normal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// If this is a selfilluminated material
	if ( globalSelfIlluminationFactor > 0.0f )
	{
		// Apply self illumination
		if ( additiveSelfIllumination )
		{
			accumulatedColour.xyz += albedo.xyz * selfIllumFactor;
		}
		else
		{
			accumulatedColour.xyz = lerp( accumulatedColour.xyz, albedo.xyz, selfIllumFactor );
		}
	}

	// Set translucency
	accumulatedColour.w = transFactor;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique HUD
<
	bool preservesGlobalState = true;
	bool supportsSpecialisedLighting = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "HUD";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx HUDVertexShader();
		PixelShader = compile sce_fp_rsx HUDFragmentShader( false, false );
#else		
		VertexShader = compile vs_3_0 HUDVertexShader();
		PixelShader = compile ps_3_0 HUDFragmentShader( false, false );
#endif
	}
}



technique HUD_Translucent
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "HUD_Translucent";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx HUDVertexShader();
		PixelShader = compile sce_fp_rsx HUDFragmentShader( true, true );
#else		
		VertexShader = compile vs_3_0 HUDVertexShader();
		PixelShader = compile ps_3_0 HUDFragmentShader( true, true );
#endif		
	}
}


technique Motec
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Motec";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx HUDVertexShader();
		PixelShader = compile sce_fp_rsx HUDFragmentShader( false, true );
#else		
		VertexShader = compile vs_3_0 HUDVertexShader();
		PixelShader = compile ps_3_0 HUDFragmentShader( false, true );
#endif
	}
}


technique HUD_Ascari_3D
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "HUD_Ascari_3D";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx HUDVertexShader();
		PixelShader = compile sce_fp_rsx HUDFragmentShader( true, true );
#else		
		VertexShader = compile vs_3_0 HUDVertexShader();
		PixelShader = compile ps_3_0 HUDFragmentShader( true, true );
#endif		
	}
}



technique HUD_Translucent_Additive
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "HUD_Translucent_Additive";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "ONE";
		string BlendOp = "ADD";	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx HUDVertexShader();
		PixelShader = compile sce_fp_rsx HUDFragmentShader( true, true );
#else		
		VertexShader = compile vs_3_0 HUDVertexShader();
		PixelShader = compile ps_3_0 HUDFragmentShader( true, true );
#endif		
	}
}
